#! /usr/bin/env python

#########################
# Script to truncate BIL files to the correct size from the associated header
# Intended for files affected by the issue with azspec adding lines to the bottom of level 1 files
#
# Author: Ben Taylor
#
# Change history
# 20 Nov 2009: Created
#########################

import sys
import optparse
import os.path
import shutil

try:    
    import data_handler
except:
    print "data_handler.py not found in library path. Please get the latest version from subversion."
    sys.exit(1)
# end try

# Command-line option handling
usage = "\n%prog -c <input_file> [<input_file> ...]\n"
usage = usage + "Fixes BIL data with extra lines on the bottom (truncates file to match header dimensions)"
parser = optparse.OptionParser(usage=usage)
parser.add_option("-c", "--copy", dest="copyfile", action="store_true", default=False, help="If set, makes a backup copy of each input file before truncating")

(options, args) = parser.parse_args()

# Check at least one input file was given
if (len(args) < 1):
    print "No input file specified"
    parser.print_help()
    sys.exit(1)
# end if

# For each file given
for filepath in args:

    # Check the file exists  
    if (not os.path.isfile(filepath)):
        print "Could not find file " + filepath
        continue
    # end if
    
    # Get the filename without path or extension
    filename = os.path.basename(filepath)
    filesplit = os.path.splitext(filename)
    filebase = filesplit[0]
    
    # See if we can find the header file to use
    if (os.path.isfile(filename + ".hdr")):
        hdrfile = filename + ".hdr"
    elif (os.path.isfile(filebase + ".hdr")):
        hdrfile = filebase + ".hdr"
    else:
        print "Could not find header file for " + filepath
        continue
    # end if       
    
    # Read the header file
    try:
        hdrdata = data_handler.readHdrFile(hdrfile)
    except:
        print "Could not read header file " + hdrfile + ". Reason: " + str(sys.exc_info()[1])
        continue
    # end try
    
    # Check the header file has appropriate dimensions
    if (not (("samples" in hdrdata) and ("lines" in hdrdata) and ("bands" in hdrdata))):
        print "Dimension info missing from header file " + hdrfile
        continue
    # end if
    
    # Work out what the new size should be
    newsize = int(hdrdata["samples"]) * int(hdrdata["lines"]) * int(hdrdata["bands"]) * 2
    
    # Create a backup copy if requested
    if (options.copyfile):
        try:
            shutil.copy(filepath, filepath + "-original")
        except:
            print "Could not make backup copy of " + filepath + ". Reason: " + str(sys.exc_info()[1])
            continue
        # end try
    # end if
    
    # Open the file in append mode (just using write mode results in it zero-filling when it's truncated)
    try:
        datafile = open(filepath, "ab")
    except:
        print "Could not open data file " + filepath + ". Reason: " + str(sys.exc_info()[1])
        continue
    # end try
    
    # Truncate the file to the right size
    try:
        datafile.truncate(newsize)
    except:
        print "Could not truncate data file " + filepath + ". Reason: " + str(sys.exc_info()[1])
        datafile.close()
        continue
    # end try
    
    # Close the file
    try:
        datafile.close()
    except:
        pass
    # end try
    
    print "Successfully truncated " + filepath
# end for

print "Finished"
